package haxe.coro;

import haxe.coro.context.Context;
import haxe.coro.context.Key;
import haxe.coro.context.IElement;
import haxe.coro.schedulers.Scheduler;
import haxe.coro.schedulers.IScheduleObject;
import haxe.CallStack.StackItem;
import haxe.Exception;

class StackTraceManager implements IElement<StackTraceManager> {
	public static final key = new Key<StackTraceManager>('StackTraceManager');

	public var insertIndex:Null<Int>;

	public function new() {

	}

	public function getKey() {
		return key;
	}
}

abstract class BaseContinuation<T> extends SuspensionResult<T> implements IContinuation<T> implements IStackFrame implements IScheduleObject {
    public final completion:IContinuation<Any>;

	public var context(get, null):Context;

    public var gotoLabel:Int;

    public var recursing:Bool;

	var resumeResult:Null<SuspensionResult<Any>>;
	#if debug
	var stackItem:Null<StackItem>;
	var startedException:Bool;
	#end

    function new(completion:IContinuation<Any>, initialLabel:Int) {
        this.completion = completion;

        gotoLabel  = initialLabel;
        error      = null;
        result     = null;
        recursing  = false;
		context    = completion.context;
		#if debug
		startedException = false;
		#end
    }

	inline function get_context() {
		return context;
	}

    public final function resume(result:Any, error:Exception):Void {
        this.result = result;
        this.error  = error;
		recursing = false;
		resumeResult = invokeResume();
		context.get(Scheduler).scheduleObject(this);
    }

    public function callerFrame():Null<IStackFrame> {
        return if (completion is IStackFrame) {
            cast completion;
        } else {
            null;
        }
    }

	public function getStackItem():Null<StackItem> {
		#if debug
		return stackItem;
		#else
		return null;
		#end
	}

    public function setClassFuncStackItem(cls:String, func:String, file:String, line:Int, pos:Int, pmin:Int, pmax:Int) {
		#if debug
        stackItem = StackItem.FilePos(StackItem.Method(cls, func), file, line, pos);
		#if eval
		eval.vm.Context.callMacroApi("associate_enum_value_pos")(stackItem, haxe.macro.Context.makePosition({file: file, min: pmin, max: pmax}));
		#end
		#end
    }

    public function setLocalFuncStackItem(id:Int, file:String, line:Int, pos:Int, pmin:Int, pmax:Int) {
		#if debug
        stackItem = StackItem.FilePos(StackItem.LocalFunction(id), file, line, pos);
		#if eval
		eval.vm.Context.callMacroApi("associate_enum_value_pos")(stackItem, haxe.macro.Context.makePosition({file: file, min: pmin, max: pmax}));
		#end
		#end
    }

	public function startException(exception:Exception) {
		#if js
		return;
		#end
		#if debug
		var stack = [];
		var skipping = 0;
		var insertIndex = 0;
		var stackItem = stackItem;
		startedException = true;

		/*
			Find first coro stack element
		*/
		while (stackItem == null) {
			var callerFrame = callerFrame();
			if (callerFrame != null) {
				stackItem = callerFrame.getStackItem();
			}
		}

		switch (stackItem) {
			case null:
				return;
			case FilePos(_, file, line, _):
				for (index => item in exception.stack.asArray()) {
					switch (item) {
						case FilePos(_, file2, line2, _) if (skipping == 0 && file == file2 && line == line2):
							stack.push(item);
							skipping = 0;
						// TODO: this is silly
						case FilePos(Method("hxcoro.CoroRun", "run"), _) if (skipping == 1):
							skipping = 2;
						// this is a hack
						case FilePos(Method(_, "invokeResume"), _) if (skipping == 0):
							skipping = 1;
							insertIndex = index;
						case _:
							if (skipping != 1) {
								stack.push(item);
							}
					}
				}
			case _:
				return;
		}
		exception.stack = stack;
		context.get(StackTraceManager).insertIndex = insertIndex;
		#end
	}

    public function buildCallStack() {
		#if js
		return;
		#end
		#if debug
		if (startedException) {
			return;
		}
		var stackTraceManager = context.get(StackTraceManager);
		// Can happen in the case of ImmediateSuspensionResult.withError
		if (stackTraceManager.insertIndex == null) {
			startException(error);
		}
		if (stackItem != null) {
			final stack = error.stack.asArray();
			stack.insert(stackTraceManager.insertIndex++, stackItem);
			error.stack = stack;
		}
		#end
    }

    abstract function invokeResume():SuspensionResult<T>;

	override function toString() {
		return '[BaseContinuation ${state.toString()}, $result]';
	}

	public function onSchedule() {
		switch (resumeResult.state) {
			case Pending:
				return;
			case Returned:
				completion.resume(resumeResult.result, null);
			case Thrown:
				completion.resume(null, resumeResult.error);
		}
	}
}